<?php

namespace Solspace\Freeform\Commands;

use craft\console\Controller;
use craft\helpers\StringHelper;
use Solspace\Freeform\Freeform;
use Solspace\Freeform\Models\FieldModel;
use yii\console\ExitCode;
use yii\helpers\Console;

class FieldsController extends Controller
{
    /**
     * @var int Number of times this field should be duplicated
     */
    public $iterations = 0;

    /**
     * @var int number to offset iterations of cloned handles and labels
     */
    public $offset = 0;

    /**
     * @var string Type of the field (text, select, checkbox, etc)
     */
    public $type = 'text';

    /**
     * @var string Instructions of the field. (optional)
     */
    public $instructions;

    /**
     * @var bool output the results without saving them in the DB
     */
    public $dryRun;

    public function options($actionID): array
    {
        return ['type', 'instructions', 'iterations', 'offset', 'dryRun'];
    }

    public function optionAliases(): array
    {
        return ['t' => 'type', 'i' => 'instructions'];
    }

    /**
     * Create a Freeform Field instance, or many duplicates of one.
     *
     * @param string $label  Label of the field
     * @param string $handle Handle of the field. (optional. will be autogenerated if left empty)
     */
    public function actionCreate(string $label, string $handle = null): int
    {
        if (empty($label)) {
            $this->stdout("Must specify a field label\n", Console::FG_YELLOW);

            return ExitCode::NOINPUT;
        }

        $this->stdout("Creating field(-s)\n", Console::FG_BLUE);

        $service = Freeform::getInstance()->fields;

        $iterations = max(0, (int) $this->iterations) + $this->offset;
        $i = max(0, (int) $this->offset);

        while ($i <= $iterations) {
            $fieldLabel = $label.($i > 0 ? ' '.$i : '');

            if ($handle) {
                $fieldHandle = $handle.($i > 0 ? '-'.$i : '');
            } else {
                $fieldHandle = StringHelper::toCamelCase($fieldLabel);
            }

            $model = new FieldModel();
            $model->type = $this->type;
            $model->label = $fieldLabel;
            $model->handle = $fieldHandle;
            $model->instructions = $this->instructions;

            $saved = $this->dryRun || $service->save($model);

            try {
                if ($saved) {
                    $this->stdout(
                        sprintf("Field #%d '%s' [%s] created.\n", $model->id, $fieldLabel, $fieldHandle),
                        Console::FG_GREEN
                    );
                } else {
                    $errors = $model->getErrorSummary(true);
                    foreach ($errors as $error) {
                        $this->stdout($error."\n", Console::FG_RED);
                    }
                }
            } catch (\Exception $exception) {
                $this->stdout(
                    $exception->getMessage()."\n",
                    Console::FG_RED
                );
            }

            ++$i;
        }

        return ExitCode::OK;
    }
}
