<?php
/**
 * CodeEditor for Craft CMS
 *
 * Provides a code editor field with Twig & Craft API autocomplete
 *
 * @link      https://nystudio107.com
 * @copyright Copyright (c) 2022 nystudio107
 */

namespace nystudio107\codeeditor;

use Craft;
use craft\events\RegisterTemplateRootsEvent;
use craft\helpers\UrlHelper;
use craft\i18n\PhpMessageSource;
use craft\web\Application as CraftWebApp;
use craft\web\View;
use nystudio107\codeeditor\helpers\Config;
use nystudio107\codeeditor\models\Settings;
use nystudio107\codeeditor\services\AutocompleteService;
use yii\base\BootstrapInterface;
use yii\base\Event;
use yii\base\Module;

/**
 * @author    nystudio107
 * @package   CodeEditor
 * @since     1.0.0
 *
 * @property AutocompleteService $autocomplete
 */
class CodeEditor extends Module implements BootstrapInterface
{
    // Constants
    // =========================================================================

    public const ID = 'codeeditor';

    public const DEFAULT_FIELD_TYPE = 'CodeEditor';

    // Public Static Properties
    // =========================================================================

    /**
     * @var Settings The CodeEditor config settings
     */
    public static $settings = null;

    // Public Methods
    // =========================================================================

    /**
     * @inerhitdoc
     */
    public function __construct($id = self::ID, $parent = null, $config = [])
    {
        /**
         * Explicitly set the $id parameter, as earlier versions of Yii2 look for a
         * default parameter, and depend on $id being explicitly set:
         * https://github.com/yiisoft/yii2/blob/f3d1534125c9c3dfe8fa65c28a4be5baa822e721/framework/di/Container.php#L436-L448
         */
        parent::__construct($id, $parent, $config);
    }

    /**
     * @inerhitDoc
     */
    public function bootstrap($app)
    {
        // Only bootstrap if this is a CraftWebApp
        if (!$app instanceof CraftWebApp) {
            return;
        }
        // Set the instance of this module class, so we can later access it with `CodeEditor::getInstance()`
        static::setInstance($this);
        // Configure our module
        $this->configureModule();
        // Register our components
        $this->registerComponents();
        // Register our event handlers
        $this->registerEventHandlers();
        Craft::info('CodeEditor module bootstrapped', __METHOD__);
    }

    // Protected Methods
    // =========================================================================

    /**
     * Configure our module
     *
     * @return void
     */
    protected function configureModule(): void
    {
        // Set up our alias
        Craft::setAlias('@nystudio107/codeeditor', $this->getBasePath());
        Craft::setAlias('@codeEditorEndpointUrl', UrlHelper::actionUrl('codeeditor/autocomplete/index'));
        // Register our module
        Craft::$app->setModule($this->id, $this);
        // Translation category
        $i18n = Craft::$app->getI18n();
        /** @noinspection UnSafeIsSetOverArrayInspection */
        if (!isset($i18n->translations[$this->id]) && !isset($i18n->translations[$this->id . '*'])) {
            $i18n->translations[$this->id] = [
                'class' => PhpMessageSource::class,
                'sourceLanguage' => 'en-US',
                'basePath' => '@nystudio107/codeeditor/translations',
                'forceTranslation' => true,
                'allowOverrides' => true,
            ];
        }
        // Set our settings
        $config = Config::getConfigFromFile($this->id);
        self::$settings = new Settings($config);
    }

    /**
     * Registers our components
     *
     * @return void
     */
    public function registerComponents(): void
    {
        $this->setComponents([
            'autocomplete' => AutocompleteService::class,
        ]);
    }

    /**
     * Registers our event handlers
     *
     * @return void
     */
    public function registerEventHandlers(): void
    {
        // Base CP templates directory
        Event::on(View::class, View::EVENT_REGISTER_CP_TEMPLATE_ROOTS, function(RegisterTemplateRootsEvent $e) {
            if (is_dir($baseDir = $this->getBasePath() . DIRECTORY_SEPARATOR . 'templates')) {
                $e->roots[$this->id] = $baseDir;
            }
        });
        // Base Site templates directory
        if (self::$settings->allowTemplateAccess) {
            Event::on(View::class, View::EVENT_REGISTER_SITE_TEMPLATE_ROOTS, function(RegisterTemplateRootsEvent $e) {
                if (is_dir($baseDir = $this->getBasePath() . DIRECTORY_SEPARATOR . 'templates')) {
                    $e->roots[$this->id] = $baseDir;
                }
            });
        }
    }
}
